#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Author : Merin Joseph

Strongly segregated polygon script uses object oriented programming to create SSPs 
that can be placed together to form any desired 2D morphology for ABC star copolymers.

"""
import matplotlib.pyplot as plt
from math import *
import numpy as np



#Characteristics of each small triangle  T_0ij
#Area0ij is the area of T_0ij
#h0ij is perpendicular height from center to the edge L0ij
#l0ij is the length of the edge
#(h_x_0ij,h_y_0ij) is the point where h0ij intersects L0ij
#t_h0ij is the distance from i vertex to the perpendicular intersection
#creating a function which takes the coordinate of the triangle to return these values


def Lengths_and_area_of_triangles(x0,y0,x_i,y_i,x_j,y_j):
        ri=[x_i,y_i]
        rj=[x_j,y_j]
        r0=[x0,y0]
        a=np.subtract(ri,r0)
        b=np.subtract(rj,r0)
        c=np.subtract(rj,ri)
        t_h_0ij=-(np.dot(a,c))/np.dot(c,c)
        #print(c)
        area0ij = 0.5*np.cross(a,b)
        if(area0ij<0):
            s=-1
        else:
            s=1
        Area0ij=abs(area0ij)
        h=np.subtract(ri,r0)+np.subtract(rj,ri)*t_h_0ij
        h0ij=np.sqrt(np.dot(h,h))
        l0ij=(sqrt((x_j-x_i)**2+(y_j-y_i)**2))        
                     
       # print(np.cross(c,d),np.cross(c,f))
       # print("t_h_0ij"+str(t_h_0ij))
        return Area0ij,h0ij,l0ij,t_h_0ij,s
       
    
    
# Streching free energy for a triangle  T_0ij    

def f_triangle_0ij(L_0ij,h_0ij,t_h,Area0ij,Area_total,phi_k,R_D):
        A=L_0ij**2
        B=h_0ij**2
        c=R_D**2  
        #t_h=sqrt((x6-rh[0])**2+(y6-rh[1])**2)
        const=(0.75*Area0ij/(Area_total*phi_k**2))
        f_ij=const*(log(c*(A*t_h**2 +B))*((A/3)*t_h**3 +B*t_h)-log(c*(A*(t_h-1)**2 +B))*((A/3)*(t_h-1)**3 +B*(t_h-1))+(2*A/9)*((t_h-1)**3-\
                    t_h**3)-(4*B/3)-(4*B/3)*(B/A)**0.5 *(atan((t_h-1)*(A/B)**0.5)-atan(t_h*(A/B)**0.5)))
        return f_ij

#interfacial energy in terms of \xi_ij=\gamma_ij*\omega/R

def f_interface(xi_ab,xi_bc,xi_ac,l0i,l0j,l0k,Area_total):
    N_chi_sst=10**(1/2)
    '''Surface energy is dependent on xi_ab,xi_bc,xi_ac'''
    f_interface=N_chi_sst*(xi_ab*l0i+xi_bc*l0j+xi_ac*l0k)/Area_total
    
    return(f_interface)
        
# create a class with object vertices.
#This will give the nodes for the polygon. Each vertice contain x and y coordinates.
 # -----------------------------------------------------------------------------------------------------
class SST_Vertex:

# Keep track of vertices as they are created
    vertex_count = 0
    vertex_list = []
    vertex_array=[]
# create a new vertex with default values
    def __init__(self):
        self.position=[0,0]
        self.id = SST_Vertex.vertex_count
        SST_Vertex.vertex_count = SST_Vertex.vertex_count + 1
        SST_Vertex.vertex_list.append(self)

# specify a string representation of a vertex
    
    def __str__(self):
        return 'Vertex %5d: (x0,y0) position (%f,%f)' % \
        (self.id, self.position[0], self.position[1])
    
    def Print(self):
        print(self)
    def clearallvertices():
        SST_Vertex.vertex_count = 0
        SST_Vertex.vertex_list=[]
        
class Centre_Vertex:

# Keep track of vertices as they are created
    centre_vertex_count = 0
    centre_vertex_list = []
    centre_vertex_array=[]
# create a new vertex with default values
    def __init__(self):
        self.position=[0,0]
        self.id = Centre_Vertex.centre_vertex_count
        Centre_Vertex.centre_vertex_count = Centre_Vertex.centre_vertex_count + 1
        Centre_Vertex.centre_vertex_list.append(self)

# specify a string representation of a vertex
    
    def __str__(self):
        return 'Vertex %5d: (x0,y0) position (%f,%f)' % \
        (self.id, self.position[0], self.position[1])
    
    def Print(self):
        print(self)
      
    def clearcentrevertex():
        Centre_Vertex.centre_vertex_count = 0
        Centre_Vertex.centre_vertex_list=[]
        
            


class Polygon():

    #make polygon with six edges and  six vertices using the previous object.
    polygon_count= 0
    polygon_list = []
    phi_a=0.3333
    phi_b=0.3333
    phi_c=1.0-phi_a-phi_b
    # xi scaling
    xi_ab=1.0
    xi_bc=1.0
    xi_ac=1.0
    R_D =10.0

# Routines to update Polygon variables    

# update the phi values (monomer compositions) of the polygon
    def update_phis(phia,phib):
        Polygon.phi_a = phia
        Polygon.phi_b = phib
        Polygon.phi_c = 1.0 - phia - phib
        if(Polygon.phi_c<=0 or Polygon.phi_b<=0 or Polygon.phi_a<=0 or 
           Polygon.phi_c>=1.0 or Polygon.phi_b>=1.0 or Polygon.phi_a>=1.0):
            print("Invalid phi values")

      # update the Flory interaction parameters xi values of the polygon  
    def update_chisR_D(chiab,chibc,chiac,RD):
        Polygon.xi_ab = chiab
        Polygon.xi_bc = chibc
        Polygon.xi_ac = chiac
        Polygon.R_D = RD
        
        
    
    # create six edges
    def __init__(self,node1,node2,node3,node4,node5,node6):
        self.vertex0=Centre_Vertex()
        self.vertex1=node1
        self.vertex2=node2
        self.vertex3=node3
        self.vertex4=node4
        self.vertex5=node5
        self.vertex6=node6
        #self.total_free_energy=0
        #self.Area_total=0
         #define the phis
        
        self.id = Polygon.polygon_count
        Polygon.polygon_count = Polygon.polygon_count + 1
        Polygon.polygon_list.append(self)  
        #self.update_phi()
        #self.update_xi()
        self.update_centre()
        self.Compute_areas_triangle()
        
        
    
    # The position of core is calculated based on the positions of the vertices and values of phi_a and phi_b.   
    def update_centre(self):
                
        
        
        #total area of polygon
        Area_total=0.5*((self.vertex1.position[0]*self.vertex2.position[1]-self.vertex2.position[0]*self.vertex1.position[1])+
                       (self.vertex2.position[0]*self.vertex3.position[1]-self.vertex3.position[0]*self.vertex2.position[1])+
                       (self.vertex3.position[0]*self.vertex4.position[1]-self.vertex4.position[0]*self.vertex3.position[1])+
                       (self.vertex4.position[0]*self.vertex5.position[1]-self.vertex5.position[0]*self.vertex4.position[1])+
                       (self.vertex5.position[0]*self.vertex6.position[1]-self.vertex6.position[0]*self.vertex5.position[1])+
                       (self.vertex6.position[0]*self.vertex1.position[1]-self.vertex1.position[0]*self.vertex6.position[1]))
        #centre vertex
        
        self.vertex0.position[0]=(((1-Polygon.phi_a-Polygon.phi_b)*2*Area_total-self.vertex3.position[0]*self.vertex4.position[1]-self.vertex4.position[0]*self.vertex5.position[1]+\
                                    self.vertex4.position[0]*self.vertex3.position[1]+self.vertex5.position[0]*self.vertex4.position[1])*(self.vertex1.position[0]-self.vertex5.position[0])- \
                                    (Polygon.phi_a*2*Area_total+self.vertex6.position[0]*self.vertex5.position[1]+self.vertex1.position[0]*self.vertex6.position[1]-self.vertex5.position[0]*self.vertex6.position[1]-\
                                     self.vertex6.position[0]*self.vertex1.position[1])*(self.vertex5.position[0]-self.vertex3.position[0]))/((self.vertex3.position[1]-\
                                     self.vertex5.position[1])*(self.vertex1.position[0]-self.vertex5.position[0])-(self.vertex5.position[1]-self.vertex1.position[1])*(self.vertex5.position[0]-self.vertex3.position[0]))
        self.vertex0.position[1]=((self.vertex4.position[0]-self.vertex6.position[0])*self.vertex5.position[1]**2 +((-self.vertex4.position[0]+\
                                   self.vertex6.position[0])*self.vertex1.position[1]+(-self.vertex4.position[0]+self.vertex6.position[0])*self.vertex3.position[1]+\
                                  (2*Polygon.phi_b-2)*Area_total+(self.vertex6.position[1]-self.vertex4.position[1])*self.vertex5.position[0]-\
                                  self.vertex6.position[1]*self.vertex1.position[0]+ self.vertex3.position[0]*self.vertex4.position[1])*self.vertex5.position[1]\
                                        +((self.vertex4.position[0]-self.vertex6.position[0])*self.vertex3.position[1] +(2-2*Polygon.phi_a-2*Polygon.phi_b)*Area_total-\
                                          self.vertex4.position[1]*(self.vertex3.position[0]-self.vertex5.position[0]))*self.vertex1.position[1]+\
                                        self.vertex3.position[1]*(2*Polygon.phi_a*Area_total+self.vertex6.position[1]*(self.vertex1.position[0]-\
                                    self.vertex5.position[0])))/((-self.vertex1.position[0]+self.vertex3.position[0])*self.vertex5.position[1]+\
                                    (self.vertex5.position[0]-self.vertex3.position[0])*self.vertex1.position[1]+self.vertex3.position[1]*(self.vertex1.position[0]- self.vertex5.position[0]))
       # print( self.vertex0.position[0],self.vertex0.position[1])
        
    def update_centre_all():
        for pg in Polygon.polygon_list:
            pg.update_centre()

    # compute areas of the six triangle
    def Compute_areas_triangle(self):
       
        # triangle 012- Btype
        x0 = self.vertex0.position[0]
        y0 = self.vertex0.position[1]
        x_i = self.vertex1.position[0]
        y_i = self.vertex1.position[1]
        x_j = self.vertex2.position[0]
        y_j = self.vertex2.position[1]
        temp012=Lengths_and_area_of_triangles(x0,y0,x_i,y_i,x_j,y_j)     
        self.Area012 = temp012[0]
        self.h012 =temp012[1]
        self.l012=temp012[2]
        self.t_h_012=temp012[3]
        self.s012=temp012[4]
        
        # triangle 023- Btype
        x_i = self.vertex2.position[0]
        y_i = self.vertex2.position[1]
        x_j = self.vertex3.position[0]
        y_j = self.vertex3.position[1]
        temp023=Lengths_and_area_of_triangles(x0,y0,x_i,y_i,x_j,y_j)       
        self.Area023 = temp023[0]
        self.h023 =temp023[1]
        self.l023=temp023[2]
        self.t_h_023=temp023[3]
        self.s023=temp023[4]
        
        
        # triangle 034- Ctype
        x_i = self.vertex3.position[0]
        y_i = self.vertex3.position[1]
        x_j = self.vertex4.position[0]
        y_j = self.vertex4.position[1]
        temp034=Lengths_and_area_of_triangles(x0,y0,x_i,y_i,x_j,y_j)       
        self.Area034 = temp034[0]
        self.h034 =temp034[1]
        self.l034=temp034[2]
        self.t_h_034=temp034[3]
        self.s034=temp034[4]
        
        # triangle 045- Ctype
        x_i = self.vertex4.position[0]
        y_i = self.vertex4.position[1]
        x_j = self.vertex5.position[0]
        y_j = self.vertex5.position[1]
        temp045=Lengths_and_area_of_triangles(x0,y0,x_i,y_i,x_j,y_j)       
        self.Area045 = temp045[0]
        self.h045 =temp045[1]
        self.l045=temp045[2]
        self.t_h_045=temp045[3]
        self.s045=temp045[4]

        # triangle 056- Atype
        x_i = self.vertex5.position[0]
        y_i = self.vertex5.position[1]
        x_j = self.vertex6.position[0]
        y_j = self.vertex6.position[1]
        temp056=Lengths_and_area_of_triangles(x0,y0,x_i,y_i,x_j,y_j)       
        self.Area056 = temp056[0]
        self.h056=temp056[1]
        self.l056=temp056[2]
        self.t_h_056=temp056[3]
        self.s056=temp056[4]

        # triangle 061- Atype
        x_i = self.vertex6.position[0]
        y_i = self.vertex6.position[1]
        x_j = self.vertex1.position[0]
        y_j = self.vertex1.position[1]
        temp061=Lengths_and_area_of_triangles(x0,y0,x_i,y_i,x_j,y_j)       
        self.Area061 = temp061[0]
        self.h061=temp061[1]
        self.l061=temp061[2]
        self.t_h_061=temp061[3]
        self.s061=temp061[4]
        
        #interface lengths
        #between AB
        self.l01=((self.vertex1.position[0]-self.vertex0.position[0])**2+(self.vertex1.position[1]-self.vertex0.position[1])**2)**(0.5)
        #between BC
        self.l03=((self.vertex3.position[0]-self.vertex0.position[0])**2+(self.vertex3.position[1]-self.vertex0.position[1])**2)**(0.5)
        #between AC
        self.l05=((self.vertex5.position[0]-self.vertex0.position[0])**2+(self.vertex5.position[1]-self.vertex0.position[1])**2)**(0.5)
        

        #total area of polygon
        self.Area_total= self.Area012 + self.Area023 + self.Area034 + self.Area045 + self.Area056 + self.Area061
        
    #Function call to update the polygon according to the values of phis and xis. This automatically updates
    #centre vertex
    #Call only this function 
    #def update_all(self,phi_a,phi_b,xi_ab,xi_bc,xi_ac,R_D):
    #    self.update_phi(phi_a,phi_b)
    #    self.update_xi(xi_ab,xi_bc,xi_ac)
    #    self.update_centre()
    #    self.Compute_free_energy(R_D)




    #Computing total free energy per chain in one polygon.
    def Compute_free_energy(self):
        self.Compute_areas_triangle()
        self.str_free_energy012=f_triangle_0ij(self.l012,self.h012,self.t_h_012,self.Area012,self.Area_total,Polygon.phi_b,Polygon.R_D)
        self.str_free_energy023=f_triangle_0ij(self.l023,self.h023,self.t_h_023,self.Area023,self.Area_total,Polygon.phi_b,Polygon.R_D)
        self.str_free_energy034=f_triangle_0ij(self.l034,self.h034,self.t_h_034,self.Area034,self.Area_total,Polygon.phi_c,Polygon.R_D)
        self.str_free_energy045=f_triangle_0ij(self.l045,self.h045,self.t_h_045,self.Area045,self.Area_total,Polygon.phi_c,Polygon.R_D)
        self.str_free_energy056=f_triangle_0ij(self.l056,self.h056,self.t_h_056,self.Area056,self.Area_total,Polygon.phi_a,Polygon.R_D)
        self.str_free_energy061=f_triangle_0ij(self.l061,self.h061,self.t_h_061,self.Area061,self.Area_total,Polygon.phi_a,Polygon.R_D)

        self.total_str_free_energy=self.str_free_energy012+self.str_free_energy023+self.str_free_energy034+\
                                                        self.str_free_energy045+self.str_free_energy056+self.str_free_energy061
        self.interfacial_free_energy=f_interface(Polygon.xi_ab,Polygon.xi_bc,Polygon.xi_ac,self.l01,self.l03,self.l05,self.Area_total)
        total_free_energy=self.total_str_free_energy+self.interfacial_free_energy
        
        #Area total calculated without including the centre
        #if(self.polygon_validity()==1)
        if(self.polygon_validity()==1):
            self.total_free_energy= total_free_energy
        else:
            Area_0=0.5*abs((self.vertex1.position[0]*self.vertex2.position[1]-self.vertex2.position[0]*self.vertex1.position[1])+
                           (self.vertex2.position[0]*self.vertex3.position[1]-self.vertex3.position[0]*self.vertex2.position[1])+
                           (self.vertex3.position[0]*self.vertex4.position[1]-self.vertex4.position[0]*self.vertex3.position[1])+
                           (self.vertex4.position[0]*self.vertex5.position[1]-self.vertex5.position[0]*self.vertex4.position[1])+
                           (self.vertex5.position[0]*self.vertex6.position[1]-self.vertex6.position[0]*self.vertex5.position[1])+
                           (self.vertex6.position[0]*self.vertex1.position[1]-self.vertex1.position[0]*self.vertex6.position[1]))
                #checking signs of triangles clockwise and anticlockwise.
            Bad_area=abs(self.Area_total-Area_0) # flag to make sure that the free energy is determined correctly during iterations.
            #print("Bad Area ",Bad_area/self.Area_total)
            self.total_free_energy= total_free_energy *(1+ ( Bad_area/self.Area_total)*100)
            
        
        
        
    

            

    
    #Check if the polygon is physically valid.
    def polygon_validity(self):
        if(Polygon.phi_c<=0):
            print("Invalid phi values")
        else:
            if((self.s012<0 and self.s023<0 and self.s034<0 and self.s045<0 and self.s056<0 and self.s061<0)or
              (self.s012>0 and self.s023>0 and self.s034>0 and self.s045>0 and self.s056>0 and self.s061>0)):                #fig,ax=pltstp.plotsetup(1,dpi=1000)
                #pltstp.axissetup(ax,1)
                #ax.set_aspect('equal')

                polygon_validity=1
        
            else:
                polygon_validity=-1
        return(polygon_validity)

        

    
    # specify a string representation of a polygon
    def __str__(self):
        return 'Polygon %3d\n vertex0 (%f,%f)\n vertex1 (%f,%f)\n vertex2 (%f,%f)\n vertex3 (%f,%f)\n vertex4 (%f,%f)\n vertex5 (%f,%f)\n vertex6 (%f,%f)\n' % \
        (self.id, self.vertex0.position[0], self.vertex0.position[1], 
                  self.vertex1.position[0], self.vertex1.position[1], 
                  self.vertex2.position[0], self.vertex2.position[1], 
                  self.vertex3.position[0], self.vertex3.position[1], 
                  self.vertex4.position[0], self.vertex4.position[1], 
                  self.vertex5.position[0], self.vertex5.position[1], 
                  self.vertex6.position[0], self.vertex6.position[1])
    def Print(self):
        print(self)
   
    def clearallpolygon():
        Polygon.polygon_count = 0
        Polygon.polygon_list=[]
        
        
    def clearall():
        SST_Vertex.clearallvertices()
        Centre_Vertex.clearcentrevertex()
        Polygon.clearallpolygon()
        
        
    #Calculates the total area of all polygons in the list. 
    def total_polygon_area():
        Atot=0.0
        for fe in Polygon.polygon_list:
            fe.update_centre()
            fe.Compute_areas_triangle()
            Atot=Atot+abs(fe.Area_total)
        return Atot

    #Calculates the free energy per area of all polygons in the list.
    def free_energy_calc():
        Atot=0.0
        FAtot=0.0
        for fe in Polygon.polygon_list:
            fe.update_centre()
            fe.Compute_free_energy()
            #print('Total free energy '+str(fe.total_free_energy),'Total area '+str(fe.Area_total))
            #print('Interfacial energy '+str(fe.interfacial_free_energy),'Stretching energy '+str(fe.total_str_free_energy))
            Atot=Atot+abs(fe.Area_total)
            FAtot=FAtot+fe.Area_total*fe.total_free_energy
        FE=FAtot/Atot
        return FE
             
            
   #Plots the polgon if it is valid.
            
    def plot_valid_polygon(ax,ssp_border=False, interface_lines=False,ssp_triangle_lines=False):
        
        
        for fe in Polygon.polygon_list:
            fe.update_centre()
            fe.Compute_areas_triangle()
            if (fe.polygon_validity()==1):
                fe.Plot(ax,ssp_border, interface_lines,ssp_triangle_lines)
                
                    
    # Plots the periodic tiling of the polygon.
    def plot_periodic_tiling(ax,r,c,X1,X2,Y2,ssp_border=False, interface_lines=False,ssp_triangle_lines=False):
        
        
        for fe in Polygon.polygon_list:
            fe.update_centre()
            fe.Compute_areas_triangle()
            if (fe.polygon_validity()==1):
                fe.Plot_periodic(ax,r,c, X1,X2,Y2,ssp_border, interface_lines,ssp_triangle_lines)
    
    def whole_polygon_validity():
        validity= 0
        for fe in Polygon.polygon_list:

            fe.update_centre()
            fe.Compute_areas_triangle()
            if (fe.polygon_validity()==1):
                validity=validity+1
            else:
                validity=-1
        if (validity==Polygon.polygon_count):
            rightpolygon=1
        else:
            rightpolygon=0
               
        return(rightpolygon)
                

    


# a routine for plotting a polygon
    def Plot(self,ax,ssp_border=False, interface_lines=False,ssp_triangle_lines=False):
        plt.gca()
        x_list = [self.vertex1.position[0], self.vertex2.position[0], self.vertex3.position[0], self.vertex4.position[0], self.vertex5.position[0], self.vertex6.position[0], self.vertex1.position[0]]
        y_list = [self.vertex1.position[1], self.vertex2.position[1], self.vertex3.position[1], self.vertex4.position[1], self.vertex5.position[1], self.vertex6.position[1], self.vertex1.position[1]]
        
        
        #Filling different polymer type regions
        A_type_x_list=[self.vertex0.position[0],self.vertex1.position[0],self.vertex6.position[0],self.vertex5.position[0],self.vertex0.position[0]]
        A_type_y_list=[self.vertex0.position[1],self.vertex1.position[1],self.vertex6.position[1],self.vertex5.position[1],self.vertex0.position[1]]

        B_type_x_list=[self.vertex0.position[0],self.vertex1.position[0],self.vertex2.position[0],self.vertex3.position[0],self.vertex0.position[0]]
        B_type_y_list=[self.vertex0.position[1],self.vertex1.position[1],self.vertex2.position[1],self.vertex3.position[1],self.vertex0.position[1]]

        C_type_x_list=[self.vertex0.position[0],self.vertex3.position[0],self.vertex4.position[0],self.vertex5.position[0],self.vertex0.position[0]]
        C_type_y_list=[self.vertex0.position[1],self.vertex3.position[1],self.vertex4.position[1],self.vertex5.position[1],self.vertex0.position[1]]


        #Sorting border lines

        x_main_1 = [self.vertex0.position[0], self.vertex1.position[0]]
        y_main_1 = [self.vertex0.position[1], self.vertex1.position[1]]
        x_main_2 = [self.vertex0.position[0], self.vertex3.position[0]]
        y_main_2 = [self.vertex0.position[1], self.vertex3.position[1]]
        x_main_3 = [self.vertex0.position[0], self.vertex5.position[0]]
        y_main_3 = [self.vertex0.position[1], self.vertex5.position[1]]

        x_mid_1 = [self.vertex0.position[0], self.vertex2.position[0]]
        y_mid_1 = [self.vertex0.position[1], self.vertex2.position[1]]
        x_mid_2 = [self.vertex0.position[0], self.vertex4.position[0]]
        y_mid_2 = [self.vertex0.position[1], self.vertex4.position[1]]
        x_mid_3 = [self.vertex0.position[0], self.vertex6.position[0]]
        y_mid_3 = [self.vertex0.position[1], self.vertex6.position[1]]
        


        # take the vertex0 to be the "centroid" of the tile
        x_centre = self.vertex0.position[0]
        y_centre = self.vertex0.position[1]
        # plot the vertices around the edge of the polygon
        if ssp_border:
            ax.plot(x_list, y_list, linestyle='solid', color='0', linewidth=0.4)
        #fill the tiles
        ax.fill(A_type_x_list,A_type_y_list,c='salmon')
        ax.fill(B_type_x_list,B_type_y_list,c='cornflowerblue')
        ax.fill(C_type_x_list,C_type_y_list,c='khaki')
        #draw the borders
        if interface_lines:
            ax.plot(x_main_1,y_main_1,x_main_2,y_main_2,x_main_3,y_main_3,color='0.4',linewidth=0.4,linestyle='solid')
        
        if ssp_triangle_lines:
            ax.plot(x_mid_1,y_mid_1,x_mid_2,y_mid_2,x_mid_3,y_mid_3,color='0.3',linewidth=0.4,linestyle='--')

    def Plot_periodic(self,ax,r,c, X1,X2,Y2,ssp_border=False, interface_lines=False,ssp_triangle_lines=False):
        plt.gca()
        '''x values periodically expanded'''
        for i in np.arange(r):
            for j in np.arange(c):
                   
                x_list = [self.vertex1.position[0]+i*X1+j*X2, self.vertex2.position[0]+i*X1+j*X2,
                          self.vertex3.position[0]+i*X1+j*X2, self.vertex4.position[0]+i*X1+j*X2, 
                          self.vertex5.position[0]+i*X1+j*X2, self.vertex6.position[0]+i*X1+j*X2, 
                          self.vertex1.position[0]+i*X1+j*X2]
                y_list = [self.vertex1.position[1]+j*Y2, self.vertex2.position[1]+j*Y2,
                          self.vertex3.position[1]+j*Y2, self.vertex4.position[1]+j*Y2, 
                          self.vertex5.position[1]+j*Y2, self.vertex6.position[1]+j*Y2, 
                          self.vertex1.position[1]+j*Y2]
                
                
                #Filling different polymer type regions
                A_type_x_list=[self.vertex0.position[0]+i*X1+j*X2,self.vertex1.position[0]+i*X1+j*X2,
                               self.vertex6.position[0]+i*X1+j*X2,self.vertex5.position[0]+i*X1+j*X2,
                               self.vertex0.position[0]+i*X1+j*X2]
                A_type_y_list=[self.vertex0.position[1]+j*Y2,self.vertex1.position[1]+j*Y2,
                               self.vertex6.position[1]+j*Y2,self.vertex5.position[1]+j*Y2,
                               self.vertex0.position[1]+j*Y2]
        
                B_type_x_list=[self.vertex0.position[0]+i*X1+j*X2,self.vertex1.position[0]+i*X1+j*X2,
                               self.vertex2.position[0]+i*X1+j*X2,self.vertex3.position[0]+i*X1+j*X2,
                               self.vertex0.position[0]+i*X1+j*X2]
                B_type_y_list=[self.vertex0.position[1]+j*Y2,self.vertex1.position[1]+j*Y2,
                               self.vertex2.position[1]+j*Y2,self.vertex3.position[1]+j*Y2,
                               self.vertex0.position[1]+j*Y2]
        
                C_type_x_list=[self.vertex0.position[0]+i*X1+j*X2,self.vertex3.position[0]+i*X1+j*X2,
                               self.vertex4.position[0]+i*X1+j*X2,self.vertex5.position[0]+i*X1+j*X2,
                               self.vertex0.position[0]+i*X1+j*X2]
                C_type_y_list=[self.vertex0.position[1]+j*Y2,self.vertex3.position[1]+j*Y2,
                               self.vertex4.position[1]+j*Y2,self.vertex5.position[1]+j*Y2,
                               self.vertex0.position[1]+j*Y2]
        
        
                #Sorting border lines
        
                x_main_1 = [self.vertex0.position[0]+i*X1+j*X2, self.vertex1.position[0]+i*X1+j*X2]
                y_main_1 = [self.vertex0.position[1]+j*Y2, self.vertex1.position[1]+j*Y2]
                x_main_2 = [self.vertex0.position[0]+i*X1+j*X2, self.vertex3.position[0]+i*X1+j*X2]
                y_main_2 = [self.vertex0.position[1]+j*Y2, self.vertex3.position[1]+j*Y2]
                x_main_3 = [self.vertex0.position[0]+i*X1+j*X2, self.vertex5.position[0]+i*X1+j*X2]
                y_main_3 = [self.vertex0.position[1]+j*Y2, self.vertex5.position[1]+j*Y2]
        
                x_mid_1 = [self.vertex0.position[0]+i*X1+j*X2, self.vertex2.position[0]+i*X1+j*X2]
                y_mid_1 = [self.vertex0.position[1]+j*Y2, self.vertex2.position[1]+j*Y2]
                x_mid_2 = [self.vertex0.position[0]+i*X1+j*X2, self.vertex4.position[0]+i*X1+j*X2]
                y_mid_2 = [self.vertex0.position[1]+j*Y2, self.vertex4.position[1]+j*Y2]
                x_mid_3 = [self.vertex0.position[0]+i*X1+j*X2, self.vertex6.position[0]+i*X1+j*X2]
                y_mid_3 = [self.vertex0.position[1]+j*Y2, self.vertex6.position[1]+j*Y2]
                
        
        
                # take the vertex0 to be the "centroid" of the tile
                x_centre = self.vertex0.position[0]+i*X1+j*X2
                y_centre = self.vertex0.position[1]+j*Y2
                # plot the vertices around the edge of the polygon
                if ssp_border:
                    ax.plot(x_list, y_list, linestyle='solid', color='#696969', linewidth=0.2)
                #fill the tiles
                ax.fill(A_type_x_list,A_type_y_list,c='salmon')
                ax.fill(B_type_x_list,B_type_y_list,c='cornflowerblue')
                ax.fill(C_type_x_list,C_type_y_list,c='khaki')
                if interface_lines:
                    ax.plot(x_main_1,y_main_1,x_main_2,y_main_2,x_main_3,y_main_3,color='0.2',linewidth=0.5,linestyle='solid')

                if ssp_triangle_lines:
                    ax.plot(x_mid_1,y_mid_1,x_mid_2,y_mid_2,x_mid_3,y_mid_3,color='0.3',linewidth=0.4,linestyle='--')


    
# Defining the periodic nodes.
# These nodes are used to apply periodicity to the vertices of the polygon.
# The periodic nodes are created based on the constraints defined by the user.      
class Periodic_nodes():

# Keep track of vertices as they are created
    periodic_node_count = 0
    periodic_node_list = []

# create a new vertex with default values
#constraint list is of the form [v1x,v2x,v2y ]
    def __init__(self,node1,node2,constraint_list,l,m):
        self.vertex1=node1
        self.vertex2=node2
        self.constraint_list=constraint_list
        self.l=l
        self.m=m
        self.id =Periodic_nodes.periodic_node_count 
        Periodic_nodes.periodic_node_count  = Periodic_nodes.periodic_node_count+ 1
        Periodic_nodes.periodic_node_list.append(self)
        
   #apply the periodicity to node     
    def periodicity_apply():
        for pn in Periodic_nodes.periodic_node_list:
            pn.vertex2.position[0]=pn.vertex1.position[0]+pn.l*pn.constraint_list[0]+pn.m*pn.constraint_list[1]
            pn.vertex2.position[1]=pn.vertex1.position[1]+pn.m*pn.constraint_list[2]
           
        
        
    def clearperiodicnodes():
        periodic_node_count = 0
        periodic_node_list = []
        
# specify a string representation of a vertex
    def __str__(self):
        return 'Periodic vertex %3d\n vertex1 (%f,%f)\n vertex2 (%f,%f)\n' % \
        (self.id, self.vertex1.position[0], self.vertex1.position[1],
                  self.vertex2.position[0], self.vertex2.position[1])
    def Print(self):
        print(self)
        
 # This class is used to create free nodes in the polygon.
# Free nodes are the vertices that are not connected to any other vertices       
class Free_nodes():
    
    free_node_count = 0
    free_node_list = []

# create a new vertex with default values
#constraint list is of the form [v1x,v2x,v2y ]
    def __init__(self,constraint_list,node):
        
        self.constraint_list=constraint_list
        self.n=len(constraint_list)
        self.vertex=node     
        constraint_list.append(node.position[0])   
        constraint_list.append(node.position[1]) 
        
        self.id =Free_nodes.free_node_count 
        Free_nodes.free_node_count   = Free_nodes.free_node_count + 1
        Free_nodes.free_node_list.append(self)
        
    def update_constraint_list():
        for fn in Free_nodes.free_node_list:
            fn.vertex.position[0]=fn.constraint_list[fn.n]
            fn.vertex.position[1]=fn.constraint_list[fn.n+1]
    def clearfreenodes():
        free_node_count = 0
        free_node_list = []
    
        
    # specify a string representation of a vertex
    def __str__(self):
        return 'Free vertex %3d\n vertex (%f,%f)\n' % \
        (self.id, self.vertex.position[0], self.vertex.position[1])
    def Print(self):
        print(self)
        
# Function to clear all the polygons, vertices and free nodes.
# This is useful to reset the simulation or to start a new simulation.
def cleareverything():
    Polygon.clearall()
    Free_nodes.clearfreenodes()
    Periodic_nodes.clearperiodicnodes()
